//----------------------------------------------------------------------------------------
// FILENAME:		FormDWAPISample.cs
//
// Copyright(c) 2009 Motorola, Inc. All rights reserved.
//
// DESCRIPTION:		This simple application demonstrates
//					how to use the Motorola DataWedge API.
//					The API's used are as follows:
//
//					Start 
//					Stop 
//					Enable 
//					Disable
//					Get Status
//					On Start Event
//					On Stop Event
//					On Data Event
//                  
//----------------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using DataWedgeLib;



namespace CSharpDWAPISample
{
    //----------------------------------------------------------------------------------------
    //
    //  FormDWAPISample: Main Form Class code
    //     
    //----------------------------------------------------------------------------------------
    public partial class FormDWAPISample : Form
    { 
        private DataWedgeAPI DWCtrl=null;
        private _IDataWedgeAPIEvents_OnStartEventHandler eventStart = null;
        private _IDataWedgeAPIEvents_OnStopEventHandler eventStop = null;
        private _IDataWedgeAPIEvents_OnDataEventHandler eventData = null;

        //----------------------------------------------------------------------------------------
        //
        //  FormDWAPISample: Main Form Class constructor
        //     
        //----------------------------------------------------------------------------------------
        public FormDWAPISample()
        {
            InitializeComponent();
            try
            {
                // Create an instance of Datawedge API Interface
                DWCtrl = new DataWedgeAPI();
                // Add OnStart Event Listener
                eventStart = new _IDataWedgeAPIEvents_OnStartEventHandler(Started);
                DWCtrl.OnStart += eventStart;
                // Add OnStop event listener
				eventStop = new _IDataWedgeAPIEvents_OnStopEventHandler(Stopped);
                DWCtrl.OnStop += eventStop;
                // Add OnData Event to handle status change events
				eventData = new _IDataWedgeAPIEvents_OnDataEventHandler(DataReceived);
                DWCtrl.OnData += eventData;
            }
            catch (Exception)
            {
                MessageBox.Show("DataWedge is not found!");
            }

        }

        //----------------------------------------------------------------------------------------
        //
        //  Started: On Start Event Handler function
        //     
        //----------------------------------------------------------------------------------------
        private void Started()
        {
            UpdateDWStatus();
        }

        //----------------------------------------------------------------------------------------
        //
        //  Stopped: On Stop Event Handler function
        //     
        //----------------------------------------------------------------------------------------
        private void Stopped()
        {
            checkBoxRunning.CheckState = CheckState.Unchecked;
        }

        private void DataReceived(string strData)
        {
            UpdateDWStatus();            
        }

        //----------------------------------------------------------------------------------------
        //
        //  UpdateDWStatus: Common function to update the State of DataWedge 
        //     
        //----------------------------------------------------------------------------------------
        private void UpdateDWStatus()
        {
            if (DWCtrl != null)
            {
                int status = DWCtrl.status;
                if (status == (int)DataWedgeLib.DataWedgeState.Stopped)
                {
                    checkBoxRunning.CheckState = CheckState.Unchecked;
                }
                else if (status == (int)DataWedgeLib.DataWedgeState.Disabled
                    || status == (int)DataWedgeLib.DataWedgeState.Sleeping
                    || status == (int)DataWedgeLib.DataWedgeState.Idle)
                {
                    checkBoxRunning.CheckState = CheckState.Indeterminate;
                }
                else
                {
                    checkBoxRunning.CheckState = CheckState.Checked;
                }
            }

        }

        //----------------------------------------------------------------------------------------
        //
        //  btnStart_Click: Start button OnClick Handler 
        //     
        //----------------------------------------------------------------------------------------
        private void btnStart_Click(object sender, EventArgs e)
        {
            if (DWCtrl == null)
                MessageBox.Show("DataWedge is not found!");
            else if (!DWCtrl.Start())
                MessageBox.Show("Start DataWedge failed.");
        }

        //----------------------------------------------------------------------------------------
        //
        //  btnStop_Click: Stop button OnClick Handler 
        //     
        //----------------------------------------------------------------------------------------
        private void btnStop_Click(object sender, EventArgs e)
        {
            if (DWCtrl == null)
                MessageBox.Show("DataWedge is not found!");
            else if (!DWCtrl.Stop())
                MessageBox.Show("Stop DataWedge failed");
        }


        //----------------------------------------------------------------------------------------
        //
        //  btnEnable_Click: Enable button OnClick Handler 
        //     
        //----------------------------------------------------------------------------------------
        private void btnEnable_Click(object sender, EventArgs e)
        {
            bool persist = true;
            if (checkBoxPersist.CheckState == CheckState.Unchecked) persist = false;

            if (DWCtrl == null)
                MessageBox.Show("DataWedge is not found!");
            else if (!DWCtrl.Enable(persist))
                MessageBox.Show("Enable DataWedge failed");
        }

        //----------------------------------------------------------------------------------------
        //
        //  btnDisable_Click: Disable button OnClick Handler 
        //     
        //----------------------------------------------------------------------------------------
        private void btnDisable_Click(object sender, EventArgs e)
        {
            bool persist = true;
            if (checkBoxPersist.CheckState == CheckState.Unchecked) persist = false;

            if (DWCtrl == null)
                MessageBox.Show("DataWedge is not found!");
            else if (!DWCtrl.Disable(persist))
                MessageBox.Show("Disable DataWedge failed");
        }

        //----------------------------------------------------------------------------------------
        //
        //  btnGetStatus_Click: Status button OnClick Handler 
        //     
        //----------------------------------------------------------------------------------------
        private void btnGetStatus_Click(object sender, EventArgs e)
        {
            string strMsg = "DataWedge is not found!";
            if (DWCtrl != null)
            {
                int status = DWCtrl.status;
                strMsg = "Not Running";
                if (status == (int)DataWedgeLib.DataWedgeState.Disabled)
                    strMsg = "Disabled";
                if (status == (int)DataWedgeLib.DataWedgeState.Idle)
                    strMsg = "No capture device";
                if (status == (int)DataWedgeLib.DataWedgeState.Sleeping)
                    strMsg = "A Profile configuration is not loaded";
                if (status == (int)DataWedgeLib.DataWedgeState.Ready)
                    strMsg = "Running";
            }
            MessageBox.Show(strMsg);

        }


        //----------------------------------------------------------------------------------------
        //
        //  FormDWAPISample_Load: Onload event handler for the FormDWAPISample 
        //     
        //----------------------------------------------------------------------------------------
        private void FormDWAPISample_Load(object sender, EventArgs e)
        {
            // Update initial state
            UpdateDWStatus();

        }

        private void FormDWAPISample_Closing(object sender, CancelEventArgs e)
        {
            ReleaseDataWedgeEvents();
        
        }

        private void btnClose_Click(object sender, EventArgs e)
        {
            ReleaseDataWedgeEvents();
            Application.Exit();
        }

        private void ReleaseDataWedgeEvents()
        {
            try
            {
                DWCtrl.OnStart -= eventStart;
                DWCtrl.OnStop -= eventStop;
                DWCtrl.OnData -= eventData;
            }
            catch (Exception) { }
            while (System.Runtime.InteropServices.Marshal.ReleaseComObject(DWCtrl) > 0) ; 
   
        }

    }
}
